/*
 * Temperatursensor-Bibliothek
 * HeatBox
 * (c) hadi-rc.de
 */
 
#define resolution 10   //Sensorauflösung in Bit (9, 10, 11 oder 12)
//-------------------------------------------------------------------------

class Tempsensor {
  private:  
    boolean sensorFound;        //Wird True wenn Sensor gefunden wurde
    boolean sensorReady;        //Wird True wenn Sensor initialisiert wurde
    boolean isInitialized;      //Wird true, wenn init() einmal aufgerufen wurde
    byte sensorAddr[8];         //64Bit Sensor-Adresse, vovon das erste Byte [Stelle0] dem SensorTyp definiert
    void initSensor();
    boolean findSensorAdress();
    boolean firstRun;            //ist TRUE wenn Sensor das erste Mal eine Messung durchführen soll. Hier warten wir 750ms vor dem Lesen der Werte.
    uint8_t pinNumber;
    OneWire* wirebus;
    uint8_t errors;
  public:
    Tempsensor(uint8_t _pin);
    int getTemp();
    uint8_t getErrors();
};

//-------------------------------------------------------------------------

 Tempsensor::Tempsensor(uint8_t _pin):     //Constructor
  sensorFound(false),
  sensorReady(false),
  isInitialized(false),
  firstRun(true),
  errors(0),
  pinNumber(_pin)
  {
    wirebus = new OneWire(pinNumber);
  }

//-------------------------------------------------------------------------

void Tempsensor::initSensor() {    //Einstellungen des Temperatur-Sensors vornehmen, z.B. die Auflösung
  wirebus->reset();
  wirebus->select(sensorAddr);     //aktuellen Sensor wählen
  wirebus->write(0x4E);            //Kommando 4E = Write ScratchPad
  wirebus->write(0x7D);            //+125°   (T_H - Register)
  wirebus->write(0xC9);            //-55.0°C (T_L - Register)
  
  switch(resolution) {
   case 10: wirebus->write(0x3F); break;    //10 Bit
   case 11: wirebus->write(0x5F); break;    //11 Bit
   case 12: wirebus->write(0x7F); break;    //12 Bit
   default: wirebus->write(0x1F); break;    // 9 Bit oder Fehlerhafte Angabe
    
  }
  //wirebus->write(0x1F);            //Configuration Register, Auflösung auf 9 Bit stellen, reicht für unsere Zwecke und höhere Auflösungen brauchen mehr Zeit (1F=9Bit 3F=10Bit 5F=11Bit 7F=12Bit)
  sensorReady=true;
}

//-------------------------------------------------------------------------

boolean Tempsensor::findSensorAdress() {                                    //Adresse des (ersten) Sensors finden
  wirebus->reset();                                                         //Bus zurücksetzen
  delay(1000);                                                              //Kurzen Augenblick warten bis der RESET stattgefunden hat
  if(!wirebus->search(sensorAddr)) {                                        //keine Adresse am BUS gefunden -> Sensor nicht/falsch angeschlossen
    errors = errors | B00000001;
    wirebus->reset_search(); 
    return false;                                                           
  }
  if(OneWire::crc8( sensorAddr, 7) != sensorAddr[7]) {                      //CRC ungültig -> Sensor fehlerhaft
    errors = errors | B00000010;
    return false; }                                                         
  if(sensorAddr[0] != 0x28 && sensorAddr[0] != 0x10) {                      //KEIN DS18B20 oder DS18S20/DS1820 - Sensor ->  28H = 18B20    10H=18S20/1820
    errors = errors | B00000100;  
    return false;                                                           
  }                             
  wirebus->reset();
  return true;                                                              //Wenn bis hierher kein Fehler passiert ist haben wir vermutlich einen Sensor gefunden
}

//-------------------------------------------------------------------------

int Tempsensor::getTemp() {                                               //Temperatur ermitteln
  if(!isInitialized){
     sensorFound=findSensorAdress();
      if(sensorFound) initSensor();
      isInitialized=true;
  }
  if(!sensorReady)return -127.0f;

  int HighByte, LowByte, SignBit, TReading, Tc_100;
  //float temp_c=0;
  wirebus->reset();
  wirebus->select(sensorAddr);
  wirebus->write(0x44,1);                                                   //Kommando 44h = Convert T (ermittle Temperatur und speichere sie im ScratchPad)
  wirebus->reset();
  if(firstRun){
     delay(750);
     firstRun=false; 
  } else {
    switch(resolution) {
     case  9: delay(94);  break; //bei 9Bit-Auflösung
     case 10: delay(188); break; //bei 10Bit-Auflösung
     case 11: delay(375); break; //bei 11Bit-Auflösung
     default: delay(750); break; //bei 12Bit-Auflösung oder Falscher Angabe
    }
  }
  wirebus->select(sensorAddr);    
  wirebus->write(0xBE);                                                     //Kommando BEh = Read ScratchPad
  
  //Das gesamte ScratchPad beinhaltet eigentlich 9 Bytes, wir brauchen nur die ersten 2.
  //Wir verzichten deshalb auf das Einlesen aller 9 Bytes mittels Schleife und sparen Zeit und RAM
  LowByte = wirebus->read();
  HighByte = wirebus->read();

  TReading = (HighByte << 8) + LowByte;
  SignBit = TReading & 0x8000;      //Prüfen ob das Byte "signiert" ist (wenn ja ist die Temperatur negativ)
  if (SignBit) TReading = -TReading;
  
  if(sensorAddr[0]==0x28){              //DS18B20
    Tc_100 = (6 * TReading) + TReading / 4;                                  //multiply by (100 * 0.0625) or 6.25
    //temp_c = (Tc_100/100) + ((float)(Tc_100 % 100)/100);                     //Der Wert in Tc_100 hat Nullen am Ende (21.5° wäre 2100.5
  } else if(sensorAddr[0]==0x10){       //DS18S20
    //temp_c = ((float)TReading/2);                                     
  }
  //return temp_c;                                                             //Ergebnis zurückgeben
  return Tc_100;
}

//-------------------------------------------------------------------------

uint8_t Tempsensor::getErrors(){
  return errors;
}

//-------------------------------------------------------------------------



